package main

import "github.com/go-gl/glfw/v3.3/glfw"
import "github.com/go-gl/gl/v4.1-core/gl"
import "github.com/go-gl/mathgl/mgl32"
import "runtime"
import "os"

import "fmt"
import "gaem"
import "math"

type ChartView struct {
	translatex, translatey float32
	translatez             float32
	roty                   float32
	rotupdown              float32
	scalex, scaley, scalez float32
}

func (c ChartView) Axes() Axis2D {
	//FIXME: This doesn't really work in 3D
	bottomleft := c.Screen2Chart(-1, -1, 0)
	bottomright := c.Screen2Chart(1, -1, 0)
	topleft := c.Screen2Chart(-1, 1, 0)
	minx := bottomleft[0]
	maxx := bottomright[0]
	if bottomright[0] < bottomleft[0] {
		minx = bottomright[0]
		maxx = bottomleft[0]
	}
	ylevel := bottomleft[1]
	if bottomright[1] < ylevel {
		ylevel = bottomright[1]
	}
	return Axis2D{minx: minx, maxx: maxx, miny: ylevel, maxy: topleft[1]}
}
func (c ChartView) GraphXfrm() mgl32.Mat4 {
	graphxfrm := mgl32.Ident4()
	graphxfrm = graphxfrm.Mul4(mgl32.HomogRotate3DX(c.rotupdown))
	graphxfrm = graphxfrm.Mul4(mgl32.HomogRotate3DY(c.roty))
	graphxfrm = graphxfrm.Mul4(mgl32.Scale3D(c.scalex, c.scaley, c.scalez))
	graphxfrm = graphxfrm.Mul4(mgl32.Translate3D(c.translatex, c.translatey, c.translatez))
	return graphxfrm
}
func (c ChartView) Screen2Chart(x, y, z float32) [3]float32 {
	rv := c.GraphXfrm().Inv().Mul4x1(mgl32.Vec4{x, y, z, 1})
	//rv := c.GraphXfrm().Mul4x1(mgl32.Vec4{x, y, z, 1})
	return [3]float32{rv.X() / rv.W(), rv.Y() / rv.W(), rv.Z() / rv.W()}
}
func (c *ChartView) Translate(x, y, z float32) {
	c.translatex += x
	c.translatey += y
	c.translatez += z
}
func (c *ChartView) TranslateScreen(x, y float32) {
	base := c.Screen2Chart(0, 0, 0)
	xvec := c.Screen2Chart(1, 0, 0)
	yvec := c.Screen2Chart(0, 1, 0)
	c.translatex += x*(xvec[0]-base[0]) + y*(yvec[0]-base[0])
	c.translatey += x*(xvec[1]-base[1]) + y*(yvec[1]-base[1])
	c.translatez += x*(xvec[2]-base[2]) + y*(yvec[2]-base[2])
}
func (c *ChartView) RotateScreen(x, y float32) {
	c.roty += x
	c.rotupdown += y
	if c.rotupdown > 1.5 {
		c.rotupdown = 1.5
	}
	if c.rotupdown < -1.5 {
		c.rotupdown = -1.5
	}
}
func (c *ChartView) Scale(x, y, z float32) {
	c.scalex *= x
	c.scaley *= y
	c.scalez *= z
}
func (c *ChartView) ScaleScreen(x, y float32) {
	//FIXME: This really doesn't work in 3D
	base := c.Screen2Chart(0, 0, 0)
	xvec := c.Screen2Chart(1, 0, 0)
	yvec := c.Screen2Chart(0, 1, 0)
	deltay := [3]float32{yvec[0] - base[0], yvec[1] - base[1], yvec[2] - base[2]}
	deltax := [3]float32{xvec[0] - base[0], xvec[1] - base[1], xvec[2] - base[2]}
	//fmt.Println("Vectors", deltax, deltay, "change", x, y)
	c.scalex = 0
	if deltax[0] != 0 {
		c.scalex = (x + 1) / deltax[0]
	}
	if deltay[1] != 0 {
		c.scaley = (y + 1) / deltay[1]
	}
	//	if deltay[0] != 0 {
	//		c.scalex += y/deltay[0]
	//	}
	//	if deltax[0] != 0 {
	//		c.scalex *= x*deltax[0]
	//	}
	//	if deltay[0] != 0 {
	//		c.scaley *= y*deltay[0]
	//	}
	//	c.scaley *= (x*deltax[1] + y*deltay[1])
	//	c.scalez *= (x*deltax[2] + y*deltay[2])
}
func NewChartView() ChartView {
	return ChartView{scalex: 1, scaley: 1, scalez: 1}
}

type DataSeries interface {
	NumValues() uint
	SetupVBO()
	VBO() uint32
}
type DataSeriesConst struct {
	value float32
	dummy DataSeriesArr
}

func (d *DataSeriesConst) NumValues() uint {
	return 0xffffffff
}
func (d *DataSeriesConst) SetupVBO() {
	d.dummy.data = []float32{d.value}
	d.dummy.SetupVBO()
}
func (d *DataSeriesConst) VBO() uint32 {
	return d.dummy.VBO()
}

type DataSeriesArr struct {
	data         []float32
	vbo          uint32
	vboGenerated bool
}

func (d *DataSeriesArr) AllValues() []float32 {
	return d.data
}
func (d *DataSeriesArr) NumValues() uint {
	return uint(len(d.data))
}
func (d *DataSeriesArr) SetupVBO() {
	if d.vboGenerated {
		//already done
		return
	}
	// Modify no particular VAO (associate later)
	gl.BindVertexArray(0)
	//create new VBO
	gl.GenBuffers(1, &d.vbo)
	gl.BindBuffer(gl.ARRAY_BUFFER, d.vbo)
	//upload data (4 bytes per value)
	toupload := d.AllValues()
	gl.BufferData(gl.ARRAY_BUFFER, len(toupload)*4, gl.Ptr(toupload), gl.STATIC_DRAW)
	//unbind VBO (for safety, in case we run something without binding in the future)
	gl.BindBuffer(gl.ARRAY_BUFFER, 0)

	d.vboGenerated = true
}
func (d *DataSeriesArr) VBO() uint32 {
	if d.vboGenerated {
		return d.vbo
	} else {
		fmt.Println("Trying to use VBO without inititalizing!")
		return 0
	}
}

type Plot struct {
	//XYZSRGBA
	Values       []string
	NumData      uint
	Style        string
	vao          uint32
	vaoGenerated bool
	instanced    bool
	startuid     uint
	Name         string
	ColorShift   [3]float32
	Hide         bool
}

var uidcount uint = 1

func (p *Plot) SetupVAO(d *map[string]DataSeries) {
	if p.vaoGenerated {
		//already done
		return
	}
	if p.Style == "Scatter" || p.Style == "Candle" {
		p.instanced = true
	}
	//Create VBOs first
	for _, value := range p.Values {
		if ds, ok := (*d)[value]; ok {
			ds.SetupVBO()
		}
	}
	gl.GenVertexArrays(1, &p.vao)
	gl.BindVertexArray(p.vao)
	for i, value := range p.Values {
		if ds, ok := (*d)[value]; ok {
			//XXX: We start attrib array locations at 1, not zero (see vertex shader layout(...))
			attribindex := i + 1
			if i == 0 || ds.NumValues() < p.NumData {
				// NumData is the size of the smallest DataSeries
				p.NumData = ds.NumValues()
			}
			gl.BindBuffer(gl.ARRAY_BUFFER, ds.VBO())
			gl.EnableVertexAttribArray(uint32(attribindex))
			//start at zero, one float per index
			gl.VertexAttribPointer(uint32(attribindex), int32(1), gl.FLOAT, false, 0, gl.Ptr(nil))
			if _, ok := ds.(*DataSeriesConst); ok {
				//all vertices, all instances get the same value (as long as we have < INT_MAX vertices)
				gl.VertexAttribDivisor(uint32(attribindex), 0xffffffff)
			} else if p.instanced {
				gl.VertexAttribDivisor(uint32(attribindex), 1)
			}
		} else {
			fmt.Println("No data for", value)
			//TODO: fake constant by setting VertexAttribDivisor to some huge number
		}
	}
	p.vaoGenerated = true
	//assign a UID to this range
	p.startuid = uidcount
	uidcount += p.NumData
}
func (p Plot) Draw() {
	if !p.vaoGenerated {
		fmt.Println("Trying to use Plot without initializing!")
		return
	}
	gl.BindVertexArray(p.vao)
	if p.Style == "Scatter" {
		gl.DrawArraysInstanced(gl.TRIANGLE_FAN, 0, 4, int32(p.NumData))
	} else if p.Style == "Candle" {
		gl.DrawArraysInstanced(gl.TRIANGLES, 0, 12, int32(p.NumData))
	} else if p.Style == "Line" {
		gl.DrawArrays(gl.LINE_STRIP, 0, int32(p.NumData))
	}

}

type Chart struct {
	Data  map[string]DataSeries
	Plots []Plot
	View  ChartView
}

func (c Chart) Draw(pipeline *gaem.Pipeline) {
	graphxfrm := c.View.GraphXfrm()
	gl.UniformMatrix4fv(pipeline.UniLoc("graphxfrm"), 1, false, &graphxfrm[0])
	for _, p := range c.Plots {
		if p.Hide {
			continue
		}
		if pipeline.MatchMtl(p.Style) {
			if loc := pipeline.UniLoc("startuid"); loc >= 0 {
				gl.Uniform1ui(loc, uint32(p.startuid))
			}
			if loc := pipeline.UniLoc("colorshift"); loc >= 0 {
				gl.Uniform3f(loc, p.ColorShift[0], p.ColorShift[1], p.ColorShift[2])
			}
			p.Draw()
		}
	}
}
func (c Chart) UidToPlotAndOffset(uid uint) (plot, offset int) {
	for i, p := range c.Plots {
		if uid >= p.startuid && uid < p.startuid+p.NumData {
			return i, int(uid - p.startuid)
		}
	}
	return -1, -1
}

func NewFakeChart() Chart {
	rv := Chart{
		Data: make(map[string]DataSeries),
		Plots: []Plot{
			Plot{
				Values:    []string{"X", "Y", "Z", "S", "R", "G", "B", "A"},
				instanced: true,
			},
		},
		View: NewChartView(),
	}
	xlist := []float32{}
	ylist := []float32{}
	zlist := []float32{}
	slist := []float32{}
	rlist := []float32{}
	glist := []float32{}
	blist := []float32{}
	alist := []float32{}
	for i := 0; i < 1000000; i++ {
		//xlist = append(xlist, float32(i)*0.01)
		xlist = append(xlist, float32(math.Mod(float64(i)*0.001, 5)))
		//ylist = append(ylist, float32(math.Abs(math.Sin(float64(i)*0.1)+0.1)))
		ylist = append(ylist, float32(math.Abs(math.Sin(float64(i)*0.005)+0.1)+float64(i/100000)*1.3))
		zlist = append(zlist, 0.000001*float32(i))
		slist = append(slist, float32(math.Mod(float64(i)*0.002, 0.01)+0.003)/4)

		rlist = append(rlist, float32(math.Abs(math.Sin(float64(i)*0.0401)+0.1)))
		glist = append(glist, float32(math.Abs(math.Sin(float64(i)*0.0502)+0.1)))
		blist = append(blist, float32(math.Abs(math.Sin(float64(i)*0.0603)+0.1)))
		alist = append(alist, 0.6)
	}
	rv.Data["X"] = &DataSeriesArr{data: xlist}
	rv.Data["Y"] = &DataSeriesArr{data: ylist}
	rv.Data["Z"] = &DataSeriesArr{data: zlist}
	rv.Data["S"] = &DataSeriesArr{data: slist}
	rv.Data["R"] = &DataSeriesArr{data: rlist}
	rv.Data["G"] = &DataSeriesArr{data: glist}
	//rv.Data["B"] = &DataSeriesArr{data: blist}
	rv.Data["B"] = &DataSeriesConst{value: 0.1}
	rv.Data["A"] = &DataSeriesArr{data: alist}

	rv.Plots[0].SetupVAO(&rv.Data)
	return rv
}

func main() {
	if len(os.Args) != 2 {
		fmt.Println("Usage: ", os.Args[0], "graphconfig.json")
		return
	}

	runtime.LockOSThread()
	glfw.Init()

	win, _ := glfw.CreateWindow(800, 600, "MOPView", nil, nil)
	win.SetInputMode(glfw.StickyKeysMode, 1) //report even very short key presses
	win.MakeContextCurrent()
	gl.Init()
	gaem.DumpErr()

	axesvao := gaem.NewVao()

	gl.Enable(gl.TEXTURE_2D)

	thepipeline, err := gaem.LoadNamedPipeline("graph")
	if err != nil {
		fmt.Println(err)
		return
	}
	glfw.SwapInterval(1)

	//fakechart := NewFakeChart()
	//fakechart, err := ChartFromConfigFile("graph.json")
	fakechart, err := ChartFromConfigFile(os.Args[1])
	if err != nil {
		fmt.Println(err)
		return
	}
	view := NewChartView()

	var startxmouse, startymouse float64
	var prevbuttonstatus glfw.Action
	var prevrightbuttonstatus glfw.Action

	win.SetScrollCallback(func(w *glfw.Window, xoff float64, yoff float64) {
		delta := float32(math.Pow(1.10, yoff))
		view.Scale(delta, delta, delta)
	})
	//infotext := ""
	movedthisclick := false
	for !win.ShouldClose() {
		//handle window resize
		dimx, dimy := win.GetFramebufferSize()
		//gl.Viewport(0, 0, int32(dimx), int32(dimy))

		//key input
		if win.GetKey(glfw.KeyW) == glfw.Press {
			view.TranslateScreen(0, -0.003)
		}
		if win.GetKey(glfw.KeyS) == glfw.Press {
			view.TranslateScreen(0, 0.003)
		}
		if win.GetKey(glfw.KeyA) == glfw.Press {
			view.TranslateScreen(0.003, 0)
		}
		if win.GetKey(glfw.KeyD) == glfw.Press {
			view.TranslateScreen(-0.003, 0)
		}
		if win.GetKey(glfw.KeyUp) == glfw.Press {
			view.Scale(1.010, 1.010, 1.010)
		}
		if win.GetKey(glfw.KeyDown) == glfw.Press {
			view.Scale(1/1.010, 1/1.010, 1/1.010)
		}
		if win.GetKey(glfw.KeyLeft) == glfw.Press {
			view.RotateScreen(0.1, 0)
		}
		if win.GetKey(glfw.KeyRight) == glfw.Press {
			view.RotateScreen(-0.1, 0)
		}
		if win.GetKey(glfw.KeyKP1) == glfw.Press {
			view.roty = 0
			view.rotupdown = 0
		}
		if win.GetKey(glfw.KeyQ) == glfw.Press {
			break
		}

		buttonstatus := win.GetMouseButton(0)
		if buttonstatus == glfw.Press && prevbuttonstatus != glfw.Press {
			startxmouse, startymouse = win.GetCursorPos()
			movedthisclick = false
		} else if buttonstatus != glfw.Press && prevbuttonstatus == glfw.Press {
			if !movedthisclick {
				//This is a click instead of drag - toggle visibility of non-clicked plots
				clicktrace := thepipeline.Fbos["staging"].ReadUint32(1)
				val := uint32(0)
				//check several points near the cursor
				offsets := [][2]int{[2]int{0, 0}, [2]int{1, 0}, [2]int{-1, 0}, [2]int{0, 1}, [2]int{0, -1}, [2]int{1, 1}, [2]int{-1, 1}, [2]int{-1, -1}, [2]int{1, -1}}
				for _, off := range offsets {
					val = clicktrace[int(startxmouse)+off[0]+(int(dimy)-int(startymouse)+off[1])*4096]
					if val != 0 {
						break
					}
				}
				if val == 0 {
					val = clicktrace[int(startxmouse+1)+(int(dimy)-int(startymouse))*4096]
				}
				if val == 0 {
					val = clicktrace[int(startxmouse)+(int(dimy)-int(startymouse+1))*4096]
				}
				plotid, offset := fakechart.UidToPlotAndOffset(uint(val))
				if plotid >= 0 {
					fmt.Println(fakechart.Plots[plotid].Name, offset)
					for i := range fakechart.Plots {
						if i == plotid {
							fakechart.Plots[i].Hide = false
						} else {
							fakechart.Plots[i].Hide = !fakechart.Plots[i].Hide
						}
					}
					//infotext = fmt.Sprintf("%s %d", fakechart.Plots[plotid].Name, offset)
				}
			}
		} else if prevbuttonstatus == glfw.Press {
			xmousecur, ymousecur := win.GetCursorPos()
			if xmousecur != startxmouse || ymousecur != startymouse {
				movedthisclick = true
			}
			view.TranslateScreen(-float32(startxmouse-xmousecur)/float32(dimx)*2, float32(startymouse-ymousecur)/float32(dimy)*2)
			startxmouse = xmousecur
			startymouse = ymousecur
		}
		prevbuttonstatus = buttonstatus
		//right mouse
		buttonstatus = win.GetMouseButton(glfw.MouseButtonRight)
		if buttonstatus == glfw.Press && prevrightbuttonstatus != glfw.Press {
			startxmouse, startymouse = win.GetCursorPos()
		} else if prevrightbuttonstatus == glfw.Press {
			xmousecur, ymousecur := win.GetCursorPos()
			deltax := -float32(startxmouse-xmousecur) / float32(dimx) * 2
			deltay := -float32(startymouse-ymousecur) / float32(dimy) * 2
			if win.GetKey(glfw.KeyLeftControl) == glfw.Press {
				view.RotateScreen(deltax, deltay)
			} else {
				view.ScaleScreen(deltax*3, -deltay*3)
				//fmt.Println(deltax, deltay)
				//fmt.Println(view.scalex, view.scaley, view.scalez)
			}
			startxmouse = xmousecur
			startymouse = ymousecur
		}
		prevrightbuttonstatus = buttonstatus

		graphxfrm := view.GraphXfrm()
		fakechart.View = view

		/////////////////
		// RENDER HERE //
		/////////////////
		thepipeline.UseStage(0)
		gl.Viewport(0, 0, int32(dimx), int32(dimy))
		if loc := thepipeline.UniLoc("screenres"); loc != -1 {
			gl.Uniform2f(loc, float32(dimx), float32(dimy))
		}
		fakechart.Draw(&thepipeline)

		thepipeline.Step()
		gl.Viewport(0, 0, int32(dimx), int32(dimy))
		if loc := thepipeline.UniLoc("screenres"); loc != -1 {
			gl.Uniform2f(loc, float32(dimx), float32(dimy))
		}
		fakechart.Draw(&thepipeline)

		thepipeline.Step()
		gl.Viewport(0, 0, int32(dimx), int32(dimy))
		if loc := thepipeline.UniLoc("screenres"); loc != -1 {
			gl.Uniform2f(loc, float32(dimx), float32(dimy))
		}
		fakechart.Draw(&thepipeline)

		thepipeline.Step()
		gl.Viewport(0, 0, int32(dimx), int32(dimy))
		gl.LineWidth(2)
		axesvao.Bind()
		gl.UniformMatrix4fv(thepipeline.UniLoc("graphxfrm"), 1, false, &graphxfrm[0])

		axes := view.Axes()
		gl.Viewport(0, 0, int32(dimx), int32(dimy))
		axesdata := axes.Lines()
		//axesdata = append(axesdata, xlatscale([3]float32{i, a.miny + vsy/20 + vsy/100, 0}, [3]float32{vsx / 40, vsy / 40, vsx / 40}, digitstring(i))...)
		axesvao.AddVbof(1, axesdata, 3)
		gl.DrawArrays(gl.LINES, 0, int32(len(axesdata)/3))

		//screen space copy
		thepipeline.Step()
		if loc := thepipeline.UniLoc("screenres"); loc != -1 {
			gl.Uniform2f(loc, float32(dimx), float32(dimy))
		}
		gl.Viewport(0, 0, int32(dimx), int32(dimy))
		gl.DrawArrays(gl.TRIANGLE_FAN, 0, int32(4))

		win.SwapBuffers()
		glfw.PollEvents()
		gaem.DumpErr()
		if win.GetKey(glfw.KeyQ) == glfw.Press {
			break
		}
	}
	fmt.Println("Exit OK")
}
