package main

import "encoding/json"
import "io/ioutil"
import "os"
import "bufio"
import "bytes"
import "fmt"
import "strconv"
import "math/rand"
import "math"

type CSVSource struct {
	Filename    string
	ColumnNames []string
	RowName     string
	SkipRows    int
	RowOffset   float32
}

func (s CSVSource) addToChart(chart *Chart) error {
	file, err := os.Open(s.Filename)
	if err != nil {
		return err
	}
	row := 0
	lineReader := bufio.NewScanner(file)
	indexes := []float32{}
	columns := make(map[string][]float32)
	for lineReader.Scan() {
		if row < s.SkipRows {
			row++
			continue
		}
		row++
		if s.RowName != "" {
			indexes = append(indexes, float32(row)+s.RowOffset)
		}
		line := lineReader.Bytes()
		parts := bytes.Split(line, []byte(","))
		for i, colname := range s.ColumnNames {
			if colname == "" {
				continue
			}
			if _, ok := columns[colname]; !ok {
				columns[colname] = make([]float32, 0, 1000) //default 1000 capacity to reduce allocations
			}
			if len(parts) < i {
				//no value this line - fill with zero
				columns[colname] = append(columns[colname], 0)
			} else if value, err := strconv.ParseFloat(string(parts[i]), 32); err == nil {
				columns[colname] = append(columns[colname], float32(value))
			} else {
				fmt.Println("Parse line", row, "col", i, "as float failed")
				columns[colname] = append(columns[colname], 0)
			}
		}
	}
	for name, array := range columns {
		if _, ok := chart.Data[name]; ok {
			fmt.Println("Multiple data sources called", name, "(ignoring definition from", s.Filename, ")")
		} else {
			chart.Data[name] = &DataSeriesArr{data: array}
		}
	}
	if s.RowName != "" {
		if _, ok := chart.Data[s.RowName]; ok {
			fmt.Println("Multiple data sources called", s.RowName, "(ignoring row name definition from", s.Filename, ")")
		} else {
			chart.Data[s.RowName] = &DataSeriesArr{data: indexes}
		}
	}
	return nil
}

type Config struct {
	CSVSources   []CSVSource
	ConstSources []struct {
		Name  string
		Value float32
	}
	Plots []struct {
		Name           string
		Style          string
		RandomizeColor float32

		X string
		Y string
		Z string
		S string
		R string
		G string
		B string
		A string
		//Candlestick stuff (Y = open)
		CY   string
		MinY string
		MaxY string
	}
}

func ChartFromConfigFile(filename string) (Chart, error) {
	rv := Chart{
		Data:  make(map[string]DataSeries),
		Plots: []Plot{},
		View:  NewChartView(),
	}
	var parsed Config
	configtext, err := ioutil.ReadFile(filename)
	if err != nil {
		return rv, err
	}
	err = json.Unmarshal(configtext, &parsed)
	if err != nil {
		return rv, err
	}
	for _, scfg := range parsed.CSVSources {
		err := scfg.addToChart(&rv)
		if err != nil {
			fmt.Println(err)
		}
	}
	for _, ccfg := range parsed.ConstSources {
		if _, ok := rv.Data[ccfg.Name]; ok {
			fmt.Println("Multiple data sources called", ccfg.Name, "(ignoring constant definition)")
		} else {
			rv.Data[ccfg.Name] = &DataSeriesConst{value: ccfg.Value}
		}
	}
	//We fill in anything not specified with "Zero" constant, so make sure it exists
	if _, ok := rv.Data["Zero"]; !ok {
		rv.Data["Zero"] = &DataSeriesConst{value: 0}
	}
	for _, plot := range parsed.Plots {
		nplot := Plot{}
		nplot.Values = append(nplot.Values, plot.X)
		nplot.Values = append(nplot.Values, plot.Y)
		nplot.Values = append(nplot.Values, plot.Z)
		nplot.Values = append(nplot.Values, plot.S)
		nplot.Values = append(nplot.Values, plot.R)
		nplot.Values = append(nplot.Values, plot.G)
		nplot.Values = append(nplot.Values, plot.B)
		nplot.Values = append(nplot.Values, plot.A)
		if plot.Style == "Candle" {
			nplot.Values = append(nplot.Values, plot.CY)
			nplot.Values = append(nplot.Values, plot.MinY)
			nplot.Values = append(nplot.Values, plot.MaxY)
		}
		for i, s := range nplot.Values {
			if s == "" {
				nplot.Values[i] = "Zero"
			}
		}
		nplot.Style = plot.Style
		nplot.SetupVAO(&rv.Data)
		nplot.Name = plot.Name
		color := rand.Float64() * 3.1416 * 2
		nplot.ColorShift[0] = float32(math.Sin(color)) * plot.RandomizeColor
		nplot.ColorShift[1] = float32(math.Sin(color+3.1416*2/3)) * plot.RandomizeColor
		nplot.ColorShift[2] = float32(math.Sin(color+3.1416*2/3*2)) * plot.RandomizeColor
		rv.Plots = append(rv.Plots, nplot)
	}
	return rv, nil
}
