package gaem

import "fmt"
import "github.com/go-gl/gl/v4.1-core/gl"

type Shader uint32

func (s Shader) Use() {
	gl.UseProgram(uint32(s))
}

func shader(source string, kind uint32, name string) (uint32, error) {
	csource, free := gl.Strs(source + "\x00")
	shad := gl.CreateShader(kind)
	gl.ShaderSource(shad, 1, csource, nil)
	free()
	gl.CompileShader(shad)
	var stat int32
	gl.GetShaderiv(shad, gl.COMPILE_STATUS, &stat)
	if stat != gl.TRUE {
		shadnam := "???"
		switch kind {
		case gl.VERTEX_SHADER:
			shadnam = "vertex"
		case gl.FRAGMENT_SHADER:
			shadnam = "fragment"
		case gl.TESS_EVALUATION_SHADER:
			shadnam = "tessellation evaluation"
		case gl.TESS_CONTROL_SHADER:
			shadnam = "tessellation control"
		case gl.GEOMETRY_SHADER:
			shadnam = "geometry"
		}
		fmt.Printf("%s shader compile failed (%s)\n", shadnam, name)
		loglen := int32(0)
		gl.GetShaderiv(shad, gl.INFO_LOG_LENGTH, &loglen)
		buf := string(make([]byte, loglen))
		gl.GetShaderInfoLog(shad, int32(len(buf)-1), nil, gl.Str(buf))
		fmt.Println(buf)
	}
	return shad, nil
}
func prog(vs, fs, name string) uint32 {
	vsi, _ := shader(vs, gl.VERTEX_SHADER, name)
	fsi, _ := shader(fs, gl.FRAGMENT_SHADER, name)
	prog := gl.CreateProgram()
	gl.AttachShader(prog, vsi)
	gl.AttachShader(prog, fsi)
	gl.LinkProgram(prog)
	var stat int32
	gl.GetProgramiv(prog, gl.LINK_STATUS, &stat)
	if stat != gl.TRUE {
		fmt.Println("link failed")
		loglen := int32(0)
		gl.GetProgramiv(prog, gl.INFO_LOG_LENGTH, &loglen)
		buf := string(make([]byte, loglen))
		gl.GetProgramInfoLog(prog, int32(len(buf)-1), nil, gl.Str(buf))
		fmt.Println(buf)
	}
	return prog
}
func NamedShader(name string) (Shader, error) {
	xsb, _ := LoadAsset("shader", "lib.xs")
	xs := string(xsb)
	vss, err := LoadAsset("shader", name+".vs")
	if err != nil {
		return 0, err
	}
	fss, err := LoadAsset("shader", name+".fs")
	if err != nil {
		return 0, err
	}
	vsi, err := shader(string(vss)+xs, gl.VERTEX_SHADER, name)
	if err != nil {
		return 0, err
	}
	fsi, err := shader(string(fss)+xs, gl.FRAGMENT_SHADER, name)
	if err != nil {
		//TODO: cleanup
		return 0, err
	}
	prog := gl.CreateProgram()
	gl.AttachShader(prog, vsi)
	gl.AttachShader(prog, fsi)

	gss, err := LoadAsset("shader", name+".gs")
	if err == nil {
		gsi, err := shader(string(gss)+xs, gl.GEOMETRY_SHADER, name)
		if err != nil {
			return 0, err
		}
		gl.AttachShader(prog, gsi)
	}
	tcss, err := LoadAsset("shader", name+".tcs")
	if err == nil {
		tsi, err := shader(string(tcss)+xs, gl.TESS_CONTROL_SHADER, name)
		if err != nil {
			return 0, err
		}
		gl.AttachShader(prog, tsi)
	}
	tess, err := LoadAsset("shader", name+".tes")
	if err == nil {
		tsi, err := shader(string(tess)+xs, gl.TESS_EVALUATION_SHADER, name)
		if err != nil {
			return 0, err
		}
		gl.AttachShader(prog, tsi)
	}

	gl.LinkProgram(prog)
	var stat int32
	gl.GetProgramiv(prog, gl.LINK_STATUS, &stat)
	if stat != gl.TRUE {
		fmt.Println("link failed")
		loglen := int32(0)
		gl.GetProgramiv(prog, gl.INFO_LOG_LENGTH, &loglen)
		buf := string(make([]byte, loglen))
		gl.GetProgramInfoLog(prog, int32(len(buf)-1), nil, gl.Str(buf))
		fmt.Println(buf)
	}
	return Shader(prog), nil

}
