package gaem

import "encoding/json"
import "fmt"
import "github.com/go-gl/gl/v4.1-core/gl"

type Pipestage struct {
	Fbo         string
	Textures    map[int]string
	FboTextures map[int]struct {
		Fbo   string
		Index int
		// May be "color", "depth" or "float"
		Kind string
		// Size to create the textures
		// Only one of these should be populated
		AbsSize []int
		//not implemented
		ScreenRelSize []float32
		LinearFilter  bool
	}
	Shader     string
	Qualmin    int
	Qualmax    int
	RenderType string
	Materials  []string
	Cullface   bool
	Clearfirst bool
}

type Pipeline struct {
	Fbos     map[string]*Fbo
	Textures map[string]texture
	Shaders  map[string]Shader
	Stages   []Pipestage
	At       int
}

func (p Pipeline) UniLoc(name string) int32 {
	DumpErr()
	rv := gl.GetUniformLocation(uint32(p.Shaders[p.Stages[p.At].Shader]), gl.Str(name+"\x00"))
	DumpErr()
	return rv
}
func (p *Pipeline) UseStage(stageindex int) {
	p.At = stageindex
	stage := p.Stages[stageindex]
	if stage.Fbo == "" {
		RenderTargetScreen()
	} else {
		p.Fbos[stage.Fbo].RenderTarget()
		gl.Viewport(0, 0, int32(p.Fbos[stage.Fbo].width), int32(p.Fbos[stage.Fbo].height))
	}
	if stage.Clearfirst {
		gl.Clear(gl.COLOR_BUFFER_BIT | gl.DEPTH_BUFFER_BIT)
	}
	for unit, name := range stage.Textures {
		p.Textures[name].BindTexToUnit(unit)
	}
	for unit, info := range stage.FboTextures {
		if info.LinearFilter {
			p.Fbos[info.Fbo].BindChannelToUnitLinear(info.Index, unit)
		} else {
			p.Fbos[info.Fbo].BindChannelToUnit(info.Index, unit)
		}
	}
	if stage.Cullface {
		gl.Enable(gl.CULL_FACE)
	} else {
		gl.Disable(gl.CULL_FACE)
	}
	p.Shaders[stage.Shader].Use()
}
func (p *Pipeline) Step() bool {
	if p.At+1 >= len(p.Stages) {
		return false
	}
	p.UseStage(p.At + 1)
	return true
}
func (p Pipeline) Current() Pipestage {
	return p.Stages[p.At]
}
func (p Pipeline) MatchMtl(mtl string) bool {
	for _, m := range p.Stages[p.At].Materials {
		if m == mtl {
			return true
		}
	}
	return false
}
func LoadNamedPipeline(name string) (Pipeline, error) {
	rv := Pipeline{}
	rv.Fbos = make(map[string]*Fbo)
	rv.Textures = make(map[string]texture)
	rv.Shaders = make(map[string]Shader)
	text, err := LoadAsset("pipeline", name+".json")
	if err != nil {
		return rv, err
	}
	err = json.Unmarshal(text, &rv.Stages)
	if err != nil {
		return rv, err
	}
	for _, stage := range rv.Stages {
		if stage.Fbo != "" {
			if _, ok := rv.Fbos[stage.Fbo]; !ok {
				// Make a new FBO
				// Size is a placeholder - will be modified by FboTextures
				fb := NewFbo(16384, 16384)
				rv.Fbos[stage.Fbo] = &fb
			}
		}
		if _, ok := rv.Shaders[stage.Shader]; !ok {
			shad, err := NamedShader(stage.Shader)
			if err != nil {
				fmt.Println(err)
			}
			rv.Shaders[stage.Shader] = shad
		}
		for _, tex := range stage.Textures {
			if _, ok := rv.Textures[tex]; !ok {
				nt, err := NamedTexture(tex)
				if err != nil {
					fmt.Println(err)
				}
				rv.Textures[tex] = nt
			}
		}
		for _, fboattach := range stage.FboTextures {
			// Add attachments to the FBO
			if fbo, ok := rv.Fbos[fboattach.Fbo]; ok {
				if _, ok := fbo.textures[fboattach.Index]; ok {
					fmt.Println("FBO", fboattach.Fbo, "texture", fboattach.Index, "used multiple times - ignoring second specification")
					continue
				}
				// Fixup size - smallest attachment determines FBO size
				if fboattach.AbsSize[0] < fbo.width {
					fbo.width = fboattach.AbsSize[0]
				}
				if fboattach.AbsSize[1] < fbo.height {
					fbo.height = fboattach.AbsSize[1]
				}
				if fboattach.Kind == "color" {
					fbo.AddColorTex(fboattach.Index)
				} else if fboattach.Kind == "float" {
					fbo.AddFloatTex(fboattach.Index)
				} else if fboattach.Kind == "uint32" {
					fbo.AddUint32Tex(fboattach.Index)
				} else if fboattach.Kind == "depth" && fboattach.Index == -1 {
					fbo.AddDepthTex()
				} else {
					fmt.Println("invalid fbo kind or depth on wrong index")
				}
				rv.Fbos[fboattach.Fbo] = fbo
			} else {
				fmt.Println("Stage wants to use FBO", stage.Fbo, "that doesn't exist! (check order)")
				fmt.Println(rv.Fbos)
			}
		}
	}
	return rv, nil
}
